--[[----------------------------------------------------------------------------

PelliExportServiceProvider.lua
Export service pour Pelli avec authentification OAuth

------------------------------------------------------------------------------]]

local LrBinding = import 'LrBinding'
local LrDialogs = import 'LrDialogs'
local LrFileUtils = import 'LrFileUtils'
local LrHttp = import 'LrHttp'
local LrPathUtils = import 'LrPathUtils'
local LrView = import 'LrView'
local LrColor = import 'LrColor'
local LrPrefs = import 'LrPrefs'
local LrTasks = import 'LrTasks'
local LrLogger = import 'LrLogger'

-- Charger le module PelliAuth depuis le plugin
local PelliAuth = dofile(_PLUGIN.path .. "/PelliAuth.lua")

-- Logger pour debug
local logger = LrLogger('PelliExport')
logger:enable('logfile')

local bind = LrView.bind
local share = LrView.share
local prefs = LrPrefs.prefsForPlugin()

local exportServiceProvider = {}

-- Fonction helper pour traduire les erreurs en messages clairs
local function getFriendlyErrorMessage(error)
	if not error then
		return "Erreur inconnue"
	end

	-- Erreurs réseau
	if error:match("Network error") or error:match("Pas de réponse") then
		return "Impossible de contacter le serveur. Vérifiez votre connexion internet."
	end

	-- Erreurs HTTP
	if error:match("HTTP 401") or error:match("invalid_token") then
		return "Session expirée. Reconnectez-vous à Pelli."
	end

	if error:match("HTTP 403") then
		return "Vous n'avez pas l'autorisation d'accéder à cette galerie."
	end

	if error:match("HTTP 404") then
		return "La galerie n'existe plus ou a été supprimée."
	end

	if error:match("HTTP 500") or error:match("HTTP 502") or error:match("HTTP 503") then
		return "Le serveur Pelli rencontre un problème temporaire. Réessayez dans quelques instants."
	end

	if error:match("timeout") or error:match("Timeout") then
		return "L'upload a pris trop de temps. Vérifiez votre connexion ou essayez avec un fichier plus petit."
	end

	-- Token
	if error:match("token") and error:match("upload") then
		return "Impossible de préparer l'upload. Reconnectez-vous à Pelli."
	end

	-- Fichier
	if error:match("Impossible de lire") then
		return "Le fichier n'a pas pu être lu. Assurez-vous qu'il est accessible."
	end

	-- Retourner le message original s'il est court et compréhensible
	if #error < 100 and not error:match("{") then
		return error
	end

	-- Message générique pour tout le reste
	return "Une erreur s'est produite lors de l'upload. Consultez les logs pour plus de détails."
end

-- Fonction helper pour encoder les URLs
local function urlEncode(str)
	if not str then return "" end
	str = string.gsub(str, "\n", "\r\n")
	str = string.gsub(str, "([^%w %-%_%.])",
		function(c) return string.format("%%%02X", string.byte(c)) end)
	str = string.gsub(str, " ", "+")
	return str
end

-- Configuration de base
exportServiceProvider.supportsIncrementalPublish = false
exportServiceProvider.canExportVideo = false
exportServiceProvider.hideSections = { 'exportLocation' }
exportServiceProvider.allowFileFormats = nil
exportServiceProvider.allowColorSpaces = nil

-- Parametres d'export
exportServiceProvider.exportPresetFields = {
	{ key = 'selectedGalleryId', default = nil },
	{ key = 'selectedGalleryName', default = "" },
	{ key = 'selectedFolderId', default = nil },
	{ key = 'selectedFolderName', default = "" },
}

-- Interface d'export
function exportServiceProvider.sectionsForTopOfDialog(f, propertyTable)
	-- Initialiser les propriétés si nécessaire
	if propertyTable.pelliAuthStatus == nil then
		propertyTable.pelliAuthStatus = PelliAuth.isAuthenticated()
	end

	if propertyTable.pelliUserName == nil then
		propertyTable.pelliUserName = prefs.pelliUserName or ""
	end

	if propertyTable.pelliUserEmail == nil then
		propertyTable.pelliUserEmail = prefs.pelliUserEmail or ""
	end

	if propertyTable.pelliStatusText == nil then
		propertyTable.pelliStatusText = propertyTable.pelliAuthStatus and "Connecté" or "Non connecté"
	end

	if propertyTable.pelliUserInfo == nil then
		local name = propertyTable.pelliUserName
		local email = propertyTable.pelliUserEmail
		if name ~= "" and email ~= "" then
			propertyTable.pelliUserInfo = name .. " (" .. email .. ")"
		else
			propertyTable.pelliUserInfo = ""
		end
	end

	-- Fonction pour charger les galeries
	local function loadGalleries()
		if not propertyTable.pelliAuthStatus then
			propertyTable.galleryChoices = {{ title = "Non connecté", value = "" }}
			return
		end

		LrTasks.startAsyncTask(function()
			local galleries, error = PelliAuth.getGalleries()

			if error or not galleries or #galleries == 0 then
				propertyTable.galleryChoices = {{ title = "Aucune galerie disponible", value = "" }}
				return
			end

			-- Créer les choix pour le popup_menu
			local choices = {}
			for i, gallery in ipairs(galleries) do
				table.insert(choices, {
					title = gallery.title,
					value = gallery.id,
				})
			end

			propertyTable.galleryChoices = choices

			-- Si une galerie était déjà sélectionnée, vérifier qu'elle existe toujours
			if propertyTable.selectedGalleryId then
				local found = false
				for _, choice in ipairs(choices) do
					if choice.value == propertyTable.selectedGalleryId then
						found = true
						break
					end
				end
				if not found then
					-- La galerie n'existe plus, réinitialiser
					propertyTable.selectedGalleryId = nil
					propertyTable.selectedGalleryName = ""
				end
			end
		end)
	end

	-- Fonction pour charger les dossiers d'une galerie
	local function loadFolders()
		if not propertyTable.selectedGalleryId or propertyTable.selectedGalleryId == "" then
			propertyTable.folderChoices = {{ title = "Racine de la galerie", value = "" }}
			return
		end

		LrTasks.startAsyncTask(function()
			local folders, error = PelliAuth.getFolders(propertyTable.selectedGalleryId)

			if error or not folders then
				propertyTable.folderChoices = {{ title = "Racine de la galerie", value = "" }}
				return
			end

			-- Créer les choix : racine + dossiers
			local choices = {{ title = "Racine de la galerie", value = "" }}
			for i, folder in ipairs(folders) do
				table.insert(choices, {
					title = folder.name,
					value = folder.id,
				})
			end

			propertyTable.folderChoices = choices
		end)
	end

	-- Initialiser les choix de galerie
	if propertyTable.galleryChoices == nil then
		propertyTable.galleryChoices = {{ title = "Chargement...", value = "" }}
		if propertyTable.pelliAuthStatus then
			loadGalleries()
		end
	end

	-- Initialiser les choix de dossier
	if propertyTable.folderChoices == nil then
		propertyTable.folderChoices = {{ title = "Racine de la galerie", value = "" }}
	end

	-- Observer pour mettre à jour automatiquement
	propertyTable:addObserver('pelliAuthStatus', function()
		propertyTable.pelliStatusText = propertyTable.pelliAuthStatus and "Connecté" or "Non connecté"
		propertyTable.pelliUserName = prefs.pelliUserName or ""
		propertyTable.pelliUserEmail = prefs.pelliUserEmail or ""

		local name = propertyTable.pelliUserName
		local email = propertyTable.pelliUserEmail
		if name ~= "" and email ~= "" then
			propertyTable.pelliUserInfo = name .. " (" .. email .. ")"
		else
			propertyTable.pelliUserInfo = ""
		end

		-- Charger les galeries quand l'utilisateur se connecte
		loadGalleries()
	end)

	-- Observer pour mettre à jour le nom de la galerie quand l'ID change
	propertyTable:addObserver('selectedGalleryId', function()
		if not propertyTable.selectedGalleryId or propertyTable.selectedGalleryId == "" then
			propertyTable.selectedGalleryName = ""
			propertyTable.selectedFolderId = nil
			propertyTable.selectedFolderName = ""
			return
		end

		-- Trouver le nom de la galerie correspondant à l'ID
		for _, choice in ipairs(propertyTable.galleryChoices or {}) do
			if choice.value == propertyTable.selectedGalleryId then
				propertyTable.selectedGalleryName = choice.title
				break
			end
		end

		-- Charger les dossiers de la galerie sélectionnée
		loadFolders()
	end)

	-- Observer pour mettre à jour le nom du dossier quand l'ID change
	propertyTable:addObserver('selectedFolderId', function()
		if not propertyTable.selectedFolderId or propertyTable.selectedFolderId == "" then
			propertyTable.selectedFolderName = ""
			return
		end

		-- Trouver le nom du dossier correspondant à l'ID
		for _, choice in ipairs(propertyTable.folderChoices or {}) do
			if choice.value == propertyTable.selectedFolderId then
				propertyTable.selectedFolderName = choice.title
				break
			end
		end
	end)

	return {
		{
			title = "Connexion Pelli",

			-- Statut de connexion
			f:row {
				f:static_text {
					title = "Statut :",
					width = share 'label_width',
				},
				f:static_text {
					title = bind 'pelliStatusText',
					text_color = bind {
						key = 'pelliAuthStatus',
						transform = function(value)
							return value and LrColor("green") or LrColor("red")
						end,
					},
				},
			},

			-- Info utilisateur (si connecté)
			f:row {
				f:static_text {
					title = "Utilisateur :",
					width = share 'label_width',
					visible = bind 'pelliAuthStatus',
				},
				f:static_text {
					title = bind 'pelliUserInfo',
					visible = bind 'pelliAuthStatus',
					fill_horizontal = 1,
				},
			},

			f:spacer { height = 10 },

			-- Boutons d'action
			f:row {
				spacing = f:control_spacing(),

				-- Bouton Se connecter (si pas connecté)
				f:push_button {
					title = "Se connecter à Pelli",
					action = function()
						LrTasks.startAsyncTask(function()
							local success = PelliAuth.authenticate()
							if success then
								-- Mettre à jour le statut d'authentification
								propertyTable.pelliAuthStatus = true
								propertyTable.pelliUserName = prefs.pelliUserName or ""
								propertyTable.pelliUserEmail = prefs.pelliUserEmail or ""
							end
						end)
					end,
					visible = bind {
						key = 'pelliAuthStatus',
						transform = function(value) return not value end,
					},
				},

				-- Bouton Se déconnecter (si connecté)
				f:push_button {
					title = "Se déconnecter",
					action = function()
						if PelliAuth.logout() then
							propertyTable.pelliAuthStatus = false
							propertyTable.selectedGalleryId = nil
							propertyTable.selectedGalleryName = ""
							propertyTable.pelliUserName = ""
							propertyTable.pelliUserEmail = ""
							propertyTable.pelliUserInfo = ""
						end
					end,
					visible = bind 'pelliAuthStatus',
				},
			},
		},

		-- Section galerie (visible seulement si authentifié)
		{
			title = "Galerie de destination",
			visible = bind 'pelliAuthStatus',

			f:row {
				f:static_text {
					title = "Galerie :",
					width = share 'label_width',
				},
				f:popup_menu {
					value = bind 'selectedGalleryId',
					items = bind 'galleryChoices',
					fill_horizontal = 1,
				},
				f:push_button {
					title = "⟳",
					tooltip = "Rafraîchir la liste des galeries",
					width = 30,
					action = function()
						loadGalleries()
					end,
				},
			},

			f:row {
				f:static_text {
					title = "Dossier :",
					width = share 'label_width',
				},
				f:popup_menu {
					value = bind 'selectedFolderId',
					items = bind 'folderChoices',
					fill_horizontal = 1,
					enabled = bind {
						key = 'selectedGalleryId',
						transform = function(value)
							return value ~= nil and value ~= ""
						end,
					},
				},
			},

			f:spacer { height = 10 },

			f:row {
				spacing = f:control_spacing(),

				f:push_button {
					title = "Créer une galerie",
					action = function()
						local galleryUrl = "https://app.pelli.io/dashboard/gallery/add"
						LrHttp.openUrlInBrowser(galleryUrl)
						LrDialogs.message(
							"Créer une galerie",
							"Une fois votre galerie créée, revenez ici et cliquez sur 'Rafraîchir' pour la sélectionner.",
							"info"
						)
					end,
				},
			},
		},
	}
end

-- Fonction pour uploader une photo vers l'API Pelli
local function uploadPhotoToAPI(filePath, fileName, uploadToken, folderId)
	-- API externe pour l'upload
	local UPLOAD_API_URL = "https://api-upload.pelli.io"

	-- Construire l'URL avec le nom du fichier, la source et optionnellement le dossier
	local uploadUrl = UPLOAD_API_URL .. "/api/photo?name=" .. urlEncode(fileName) .. "&source=lightroom"
	if folderId and folderId ~= "" then
		uploadUrl = uploadUrl .. "&folder=" .. folderId
	end

	logger:info('=== Upload Photo to API ===')
	logger:info('URL: ' .. uploadUrl)
	logger:info('File: ' .. fileName)
	logger:info('Path: ' .. filePath)
	logger:info('Token length: ' .. tostring(#uploadToken))
	logger:info('Token first 20 chars: ' .. uploadToken:sub(1, 20) .. '...')

	-- Lire le fichier
	local fileContent = LrFileUtils.readFile(filePath)
	if not fileContent then
		logger:error('Failed to read file: ' .. filePath)
		return false, "Impossible de lire le fichier"
	end

	logger:info('File size: ' .. tostring(#fileContent) .. ' bytes')

	-- Générer une boundary unique pour multipart/form-data
	local boundary = "----LightroomBoundary" .. os.time() .. math.random(1000000, 9999999)

	-- Construire le corps multipart/form-data
	local body = ""
	body = body .. "--" .. boundary .. "\r\n"
	body = body .. 'Content-Disposition: form-data; name="file"; filename="' .. fileName .. '"\r\n'
	body = body .. "Content-Type: application/octet-stream\r\n"
	body = body .. "\r\n"
	body = body .. fileContent
	body = body .. "\r\n--" .. boundary .. "--\r\n"

	-- Headers HTTP
	local headers = {
		{ field = "Content-Type", value = "multipart/form-data; boundary=" .. boundary },
		{ field = "Authorization", value = uploadToken },
		{ field = "Content-Length", value = tostring(#body) },
	}

	-- Envoyer la requête POST
	logger:info('Sending POST request...')
	logger:info('Body size: ' .. tostring(#body) .. ' bytes')
	logger:info('Boundary: ' .. boundary)

	local response, hdrs = LrHttp.post(uploadUrl, body, headers, "POST", 300) -- 5 min timeout

	-- Log de la réponse
	logger:info('=== Upload Response ===')
	if hdrs and hdrs.status then
		logger:info('HTTP Status: ' .. tostring(hdrs.status))
	else
		logger:error('No HTTP headers received')
	end

	if response then
		logger:info('Response body: ' .. tostring(response))
	else
		logger:error('No response body received')
	end

	-- Vérifier le succès
	if hdrs and hdrs.status then
		if hdrs.status >= 200 and hdrs.status < 300 then
			logger:info('Upload successful!')
			return true, nil
		else
			logger:error('Upload failed with status ' .. hdrs.status)
			return false, "Erreur HTTP " .. hdrs.status .. ": " .. (response or "")
		end
	else
		logger:error('No server response')
		return false, "Pas de réponse du serveur"
	end
end

-- Processus d'export
function exportServiceProvider.processRenderedPhotos(functionContext, exportContext)
	local exportSession = exportContext.exportSession
	local exportSettings = exportContext.propertyTable
	local nPhotos = exportSession:countRenditions()

	-- Vérifier que l'utilisateur est connecté
	if not PelliAuth.isAuthenticated() then
		LrDialogs.message(
			"Erreur d'export",
			"Vous devez vous connecter à Pelli avant d'exporter.",
			"critical"
		)
		return
	end

	-- Vérifier que le token est encore valide auprès du serveur
	local tokenValid, tokenValidError = PelliAuth.verifyToken()
	if not tokenValid then
		if tokenValidError == "expired" then
			LrDialogs.message(
				"Session expirée",
				"Votre session Pelli a expiré. Veuillez vous reconnecter depuis les paramètres d'export.",
				"critical"
			)
		else
			LrDialogs.message(
				"Erreur de connexion",
				"Impossible de vérifier votre session Pelli. Vérifiez votre connexion internet.",
				"critical"
			)
		end
		return
	end

	-- Vérifier qu'une galerie est sélectionnée
	if not exportSettings.selectedGalleryId or exportSettings.selectedGalleryId == "" then
		LrDialogs.message(
			"Erreur d'export",
			"Vous devez sélectionner une galerie de destination.",
			"critical"
		)
		return
	end

	local galleryId = exportSettings.selectedGalleryId
	local galleryName = exportSettings.selectedGalleryName or "Unknown Gallery"
	local folderId = exportSettings.selectedFolderId

	-- Obtenir le token upload pour cette galerie
	local uploadToken, tokenError = PelliAuth.getUploadToken(galleryId)
	if tokenError or not uploadToken then
		LrDialogs.message(
			"Erreur d'export",
			"Impossible d'obtenir le token d'upload : " .. (tokenError or "Unknown error"),
			"critical"
		)
		return
	end

	local progressScope = exportContext:configureProgress {
		title = "Upload vers Pelli",
	}

	local successCount = 0
	local failureCount = 0
	local errorMessages = {}

	for i, rendition in exportContext:renditions() do
		if progressScope:isCanceled() then
			break
		end

		local photo = rendition.photo
		local originalPhotoName = photo:getFormattedMetadata('fileName') or "Unknown"

		progressScope:setPortionComplete(i - 1, nPhotos)
		progressScope:setCaption("Upload de " .. originalPhotoName)

		local successRender, pathOrMessage = rendition:waitForRender()

		if successRender then
			-- Utiliser le nom du fichier RENDU (avec la bonne extension .jpg), pas l'original
			local renderedFileName = LrPathUtils.leafName(pathOrMessage)

			-- Uploader la photo vers l'API Pelli (avec retry automatique)
			local uploadSuccess, uploadError
			local maxRetries = 3

			for attempt = 1, maxRetries do
				uploadSuccess, uploadError = uploadPhotoToAPI(
					pathOrMessage,
					renderedFileName,
					uploadToken,
					folderId
				)

				if uploadSuccess then
					break
				end

				-- Ne pas retenter les erreurs client (authentification, permission, ressource introuvable)
				if uploadError and (uploadError:match("Erreur HTTP 401") or uploadError:match("Erreur HTTP 403") or uploadError:match("Erreur HTTP 404")) then
					logger:info('Erreur client, pas de retry: ' .. tostring(uploadError))
					break
				end

				if attempt < maxRetries then
					local delay = attempt * 2
					logger:info('Retry upload ' .. renderedFileName .. ' (' .. attempt .. '/' .. maxRetries .. ') dans ' .. delay .. 's...')
					progressScope:setCaption("Retry " .. renderedFileName .. " (" .. attempt .. "/" .. maxRetries .. ")")
					LrTasks.sleep(delay)
				end
			end

			if uploadSuccess then
				successCount = successCount + 1
			else
				failureCount = failureCount + 1
				local friendlyError = getFriendlyErrorMessage(uploadError)
				table.insert(errorMessages, renderedFileName .. ": " .. friendlyError)
			end

			-- Supprimer le fichier temporaire
			LrFileUtils.delete(pathOrMessage)
		else
			failureCount = failureCount + 1
			table.insert(errorMessages, originalPhotoName .. ": Le rendu de l'image a échoué")
		end
	end

	progressScope:done()

	if failureCount > 0 then
		local errorDetail = ""
		if #errorMessages > 0 and #errorMessages <= 5 then
			errorDetail = "\n\nErreurs:\n" .. table.concat(errorMessages, "\n")
		elseif #errorMessages > 5 then
			-- Afficher les 5 premières erreurs
			local firstErrors = {}
			for i = 1, 5 do
				table.insert(firstErrors, errorMessages[i])
			end
			errorDetail = "\n\nPremières erreurs:\n" .. table.concat(firstErrors, "\n") .. "\n... et " .. (#errorMessages - 5) .. " autres"
		end

		LrDialogs.message(
			"Upload Pelli terminé avec erreurs",
			successCount .. " photos uploadées avec succès, " .. failureCount .. " échecs" .. errorDetail,
			"warning"
		)
	else
		LrDialogs.message(
			"Upload Pelli réussi",
			successCount .. " photos uploadées avec succès vers la galerie '" .. galleryName .. "'",
			"info"
		)
	end
end

return exportServiceProvider
