--[[----------------------------------------------------------------------------

PelliAuth.lua
Gestion de l'authentification simplifiée pour Pelli

------------------------------------------------------------------------------]]

local LrDialogs = import 'LrDialogs'
local LrHttp = import 'LrHttp'
local LrPrefs = import 'LrPrefs'
local LrTasks = import 'LrTasks'
local LrFunctionContext = import 'LrFunctionContext'
local LrView = import 'LrView'
local LrBinding = import 'LrBinding'
local LrColor = import 'LrColor'
local LrLogger = import 'LrLogger'

local bind = LrView.bind
local share = LrView.share

-- Activer les logs
local logger = LrLogger('PelliAuth')
logger:enable('logfile')

local PelliAuth = {}

-- Configuration
local API_URL = "https://app.pelli.io/api/lightroom"
local WEB_URL = "https://app.pelli.io"

-- Preferences
local prefs = LrPrefs.prefsForPlugin()

--============================================================================--
-- Simple JSON encoder/decoder
--============================================================================--

local function jsonEncode(obj)
	local function escape(s)
		s = tostring(s)
		s = s:gsub('\\', '\\\\')
		s = s:gsub('"', '\\"')
		s = s:gsub('\n', '\\n')
		return s
	end

	if type(obj) == 'table' then
		local parts = {}
		for k, v in pairs(obj) do
			if type(k) == 'string' then
				local value
				if type(v) == 'string' then
					value = '"' .. escape(v) .. '"'
				elseif type(v) == 'number' or type(v) == 'boolean' then
					value = tostring(v)
				else
					value = 'null'
				end
				table.insert(parts, '"' .. escape(k) .. '":' .. value)
			end
		end
		return '{' .. table.concat(parts, ',') .. '}'
	else
		return '""'
	end
end

local function jsonDecode(jsonString)
	if not jsonString or jsonString == "" then
		return {}
	end

	-- Simple parser qui gère les tableaux et objets
	local pos = 1

	local function skipWhitespace()
		while pos <= #jsonString do
			local c = jsonString:sub(pos, pos)
			if c ~= ' ' and c ~= '\t' and c ~= '\n' and c ~= '\r' then
				break
			end
			pos = pos + 1
		end
	end

	local function parseString()
		pos = pos + 1 -- skip opening quote
		local parts = {}
		while pos <= #jsonString do
			local c = jsonString:sub(pos, pos)
			if c == '\\' then
				pos = pos + 1
				local nc = jsonString:sub(pos, pos)
				if nc == '"' then table.insert(parts, '"')
				elseif nc == '\\' then table.insert(parts, '\\')
				elseif nc == '/' then table.insert(parts, '/')
				elseif nc == 'n' then table.insert(parts, '\n')
				elseif nc == 't' then table.insert(parts, '\t')
				elseif nc == 'r' then table.insert(parts, '\r')
				elseif nc == 'b' then table.insert(parts, '\b')
				elseif nc == 'f' then table.insert(parts, '\f')
				elseif nc == 'u' then
					local hex = jsonString:sub(pos + 1, pos + 4)
					local codepoint = tonumber(hex, 16)
					pos = pos + 4
					if codepoint then
						if codepoint < 128 then
							table.insert(parts, string.char(codepoint))
						elseif codepoint < 2048 then
							table.insert(parts, string.char(
								192 + math.floor(codepoint / 64),
								128 + (codepoint % 64)
							))
						elseif codepoint < 65536 then
							table.insert(parts, string.char(
								224 + math.floor(codepoint / 4096),
								128 + math.floor((codepoint % 4096) / 64),
								128 + (codepoint % 64)
							))
						end
					end
				else
					table.insert(parts, nc)
				end
			elseif c == '"' then
				pos = pos + 1
				return table.concat(parts)
			else
				table.insert(parts, c)
			end
			pos = pos + 1
		end
		return table.concat(parts)
	end

	local function parseValue()
		skipWhitespace()
		local c = jsonString:sub(pos, pos)

		-- String
		if c == '"' then
			return parseString()
		end

		-- Array
		if c == '[' then
			pos = pos + 1
			local array = {}
			skipWhitespace()
			if jsonString:sub(pos, pos) == ']' then
				pos = pos + 1
				return array
			end

			while true do
				local value = parseValue()
				table.insert(array, value)
				skipWhitespace()
				c = jsonString:sub(pos, pos)
				if c == ']' then
					pos = pos + 1
					break
				elseif c == ',' then
					pos = pos + 1
				end
			end
			return array
		end

		-- Object
		if c == '{' then
			pos = pos + 1
			local obj = {}
			skipWhitespace()
			if jsonString:sub(pos, pos) == '}' then
				pos = pos + 1
				return obj
			end

			while true do
				skipWhitespace()
				-- Parse key
				if jsonString:sub(pos, pos) ~= '"' then break end
				local key = parseString()

				skipWhitespace()
				if jsonString:sub(pos, pos) == ':' then
					pos = pos + 1
				end

				-- Parse value
				local value = parseValue()
				obj[key] = value

				skipWhitespace()
				c = jsonString:sub(pos, pos)
				if c == '}' then
					pos = pos + 1
					break
				elseif c == ',' then
					pos = pos + 1
				end
			end
			return obj
		end

		-- Number
		if c == '-' or (c >= '0' and c <= '9') then
			local startPos = pos
			if c == '-' then pos = pos + 1 end
			while pos <= #jsonString do
				c = jsonString:sub(pos, pos)
				if not ((c >= '0' and c <= '9') or c == '.') then
					break
				end
				pos = pos + 1
			end
			return tonumber(jsonString:sub(startPos, pos - 1))
		end

		-- Boolean / null
		if jsonString:sub(pos, pos + 3) == 'true' then
			pos = pos + 4
			return true
		elseif jsonString:sub(pos, pos + 4) == 'false' then
			pos = pos + 5
			return false
		elseif jsonString:sub(pos, pos + 3) == 'null' then
			pos = pos + 4
			return nil
		end

		return nil
	end

	return parseValue()
end

--============================================================================--
-- Helpers
--============================================================================--

-- Générer un session ID unique
local function generateSessionId()
	return "lr_" ..
		os.time() .. "_" ..
		math.random(100000, 999999)
end

local function makeRequest(method, endpoint, body, token)
	local url = API_URL .. endpoint
	local headers = {
		{ field = 'Content-Type', value = 'application/json' },
	}

	if token then
		table.insert(headers, { field = 'Authorization', value = 'Bearer ' .. token })
	end

	local jsonBody = body and jsonEncode(body) or nil

	logger:trace('Making ' .. method .. ' request to: ' .. url)
	if jsonBody then
		logger:trace('Request body: ' .. jsonBody)
	end

	local response, hdrs
	if method == "POST" then
		-- Le body doit être du JSON valide, envoyer {} si pas de body
		local postBody = jsonBody or "{}"
		response, hdrs = LrHttp.post(url, postBody, headers)
	else
		response, hdrs = LrHttp.get(url, headers)
	end

	-- Log response status
	if hdrs and hdrs.status then
		logger:trace('Response status: ' .. tostring(hdrs.status))
	else
		logger:error('No HTTP response headers received from: ' .. url)
	end

	if response then
		local success, data = pcall(jsonDecode, response)
		if not success then
			logger:error('JSON decode error: ' .. tostring(data))
			return { error = "JSON decode error: " .. tostring(data) }, hdrs
		end

		-- Vérifier que data est une table (objet JSON)
		if type(data) ~= "table" then
			logger:error('JSON decode returned non-table type: ' .. type(data) .. ', value: ' .. tostring(data))
			return { error = "Invalid JSON response type: " .. type(data) }, hdrs
		end

		return data, hdrs
	else
		-- Log l'erreur complète
		logger:error('Network error - No response from: ' .. url)
		return { error = "Network error: " .. url }, nil
	end
end

--============================================================================--
-- Token Management
--============================================================================--

function PelliAuth.saveToken(accessToken, userId, userName, userEmail)
	prefs.pelliAccessToken = accessToken
	prefs.pelliUserId = userId
	prefs.pelliUserName = userName or ""
	prefs.pelliUserEmail = userEmail or ""
	prefs.pelliTokenDate = os.time()
end

function PelliAuth.getToken()
	return prefs.pelliAccessToken
end

function PelliAuth.getUserId()
	return prefs.pelliUserId
end

function PelliAuth.clearToken()
	prefs.pelliAccessToken = nil
	prefs.pelliUserId = nil
	prefs.pelliTokenDate = nil
	prefs.pelliUserName = nil
	prefs.pelliUserEmail = nil
end

function PelliAuth.isAuthenticated()
	return prefs.pelliAccessToken ~= nil
end

function PelliAuth.verifyToken()
	local token = PelliAuth.getToken()
	if not token then
		return false, "no_token"
	end

	local data, hdrs = makeRequest('GET', '/verify', nil, token)

	if hdrs and hdrs.status == 401 then
		logger:info('Token expired or invalid, clearing')
		PelliAuth.clearToken()
		return false, "expired"
	end

	if not hdrs then
		return false, "network"
	end

	return true
end

--============================================================================--
-- Simplified Authentication Flow
--============================================================================--

function PelliAuth.authenticate()
	return LrFunctionContext.callWithContext('PelliAuth', function(context)
		-- Générer un session ID unique
		local sessionId = generateSessionId()
		logger:trace('Starting auth with session ID: ' .. sessionId)

		-- Créer la session sur le serveur
		local sessionData, _ = makeRequest('POST', '/auth/create-session', { sessionId = sessionId }, nil)

		if sessionData.error then
			LrDialogs.message(
				"Erreur",
				"Impossible de créer la session d'authentification.",
				"critical"
			)
			return false
		end

		-- Ouvrir le navigateur automatiquement
		local authUrl = WEB_URL .. "/lightroom/auth?session=" .. sessionId
		logger:trace('Opening browser to: ' .. authUrl)
		LrHttp.openUrlInBrowser(authUrl)

		-- Afficher une barre de progression
		local progressScope = LrDialogs.showModalProgressDialog({
			title = "Authentification Pelli",
			caption = "En attente de l'autorisation dans le navigateur...",
			functionContext = context,
			cannotCancel = false,
		})

		-- Polling pour vérifier si l'utilisateur a approuvé
		local maxAttempts = 300 -- 300 * 2 secondes = 10 minutes max
		local attempt = 0
		local authenticated = false
		local tokenData = nil

		logger:trace('Starting polling for session approval...')

		while attempt < maxAttempts do
			-- Vérifier si l'utilisateur a annulé
			if progressScope:isCanceled() then
				logger:info('Authentication canceled by user')
				break
			end

			-- Mettre à jour la progression
			progressScope:setPortionComplete(attempt, maxAttempts)
			local timeRemaining = math.ceil((maxAttempts - attempt) * 2 / 60)
			progressScope:setCaption("En attente de l'autorisation... (" .. timeRemaining .. " min restantes)")

			LrTasks.sleep(2) -- Attendre 2 secondes entre chaque vérification

			-- Vérifier le statut de la session
			local statusData, _ = makeRequest('POST', '/auth/check-session', { sessionId = sessionId }, nil)

			if statusData.approved and statusData.token then
				-- Authentification réussie!
				tokenData = statusData
				authenticated = true
				logger:info('Authentication successful')
				break
			elseif statusData.error == "session_expired" then
				-- Session expirée
				logger:error('Session expired')
				break
			end

			attempt = attempt + 1
			logger:trace('Polling attempt ' .. attempt .. '/' .. maxAttempts)
		end

		progressScope:done()

		if authenticated and tokenData then
			-- Sauvegarder le token
			PelliAuth.saveToken(
				tokenData.token,
				tokenData.userId,
				tokenData.userName,
				tokenData.userEmail
			)

			LrDialogs.message(
				"Authentification réussie",
				"Vous êtes maintenant connecté à Pelli en tant que " .. (tokenData.userName or tokenData.userEmail) .. ".",
				"info"
			)

			return true
		else
			-- Vérifier si l'utilisateur a annulé
			if progressScope:isCanceled() then
				LrDialogs.message(
					"Authentification annulée",
					"L'authentification a été annulée.",
					"info"
				)
			else
				LrDialogs.message(
					"Délai dépassé",
					"L'authentification a expiré après 10 minutes. Veuillez réessayer.",
					"warning"
				)
			end
			return false
		end
	end)
end

--============================================================================--
-- Gallery Selection
--============================================================================--

function PelliAuth.getGalleries()
	local token = PelliAuth.getToken()
	if not token then
		return nil, "Not authenticated"
	end

	local data, hdrs = makeRequest('POST', '/galleries/list', {}, token)

	if data.error then
		return nil, data.error
	end

	return data.galleries
end

function PelliAuth.getFolders(galleryId)
	local token = PelliAuth.getToken()
	if not token then
		return nil, "Not authenticated"
	end

	local data, hdrs = makeRequest('POST', '/galleries/folders', { galleryId = galleryId }, token)

	if data.error then
		return nil, data.error
	end

	return data.folders
end

function PelliAuth.getUploadToken(galleryId)
	local token = PelliAuth.getToken()
	if not token then
		return nil, "Not authenticated"
	end

	local data, hdrs = makeRequest('POST', '/upload-token', { galleryId = galleryId }, token)

	-- Debug logging
	logger:trace("Upload token response type: " .. type(data))
	if hdrs and hdrs.status then
		logger:trace("Upload token HTTP status: " .. hdrs.status)
	end

	-- Vérifier que data est une table
	if type(data) ~= "table" then
		logger:error("Upload token response is not a table, got: " .. tostring(data))
		return nil, "Réponse invalide du serveur (type: " .. type(data) .. ")"
	end

	if data.error then
		return nil, data.error
	end

	if not data.uploadToken then
		logger:error("No uploadToken in response")
		return nil, "Pas de token d'upload dans la réponse"
	end

	return data.uploadToken
end

function PelliAuth.selectGallery(propertyTable)
	return LrFunctionContext.callWithContext('selectGallery', function(context)
		-- Récupérer les galeries
		local galleries, error = PelliAuth.getGalleries()

		if error then
			LrDialogs.message(
				"Erreur",
				"Impossible de récupérer les galeries : " .. error,
				"critical"
			)
			return
		end

		if not galleries or #galleries == 0 then
			LrDialogs.message(
				"Aucune galerie",
				"Vous n'avez pas encore de galerie. Cliquez sur 'Créer une galerie' pour en créer une.",
				"info"
			)
			return
		end

		-- Créer une observable property pour la sélection
		local props = LrBinding.makePropertyTable(context)
		props.selectedGalleryIndex = 1  -- Par défaut : premier choix

		-- Afficher le dialogue de sélection avec radio buttons
		local f = LrView.osFactory()

		-- Créer les radio buttons pour chaque galerie
		local radioButtons = {}
		for i, gallery in ipairs(galleries) do
			table.insert(radioButtons, f:radio_button {
				title = gallery.title,
				value = i,
				checked_value = bind 'selectedGalleryIndex',
			})
		end

		local result = LrDialogs.presentModalDialog {
			title = "Choisir une galerie",
			contents = f:column {
				spacing = f:control_spacing(),

				f:static_text {
					title = "Sélectionnez la galerie de destination :",
				},

				f:scrolled_view {
					width = 400,
					height = 300,
					f:column(radioButtons),
				},
			},
			actionVerb = "Choisir",
			bindToObject = props,
		}

		if result == 'ok' then
			local selectedIndex = props.selectedGalleryIndex
			local selectedGallery = galleries[selectedIndex]

			if selectedGallery then
				propertyTable.selectedGalleryId = selectedGallery.id
				propertyTable.selectedGalleryName = selectedGallery.title
				logger:trace('Gallery selected: ' .. selectedGallery.title .. ' (ID: ' .. selectedGallery.id .. ')')

				LrDialogs.message(
					"Galerie sélectionnée",
					"Galerie : " .. selectedGallery.title,
					"info"
				)
			else
				logger:error('ERROR: No gallery found at index: ' .. tostring(selectedIndex))
				LrDialogs.message(
					"Erreur",
					"Impossible de trouver la galerie sélectionnée (index invalide).",
					"warning"
				)
			end
		end
	end)
end

--============================================================================--
-- Logout
--============================================================================--

function PelliAuth.logout()
	PelliAuth.clearToken()
	LrDialogs.message("Déconnexion réussie", "Vous avez été déconnecté de Pelli.", "info")
	return true
end

return PelliAuth
